import React, { useState, useEffect, useRef } from 'react';
import { textColor, onlyColor } from '../constant';


//style classes
const cardClass = "bg-muted p-4 rounded-lg";
const textClass = "text-muted-foreground font-bold text-zinc-700";
const textClasses = "text-sm text-zinc-500";
const fillClass = `text-3xl ${textColor} font-semibold text-secondary`;
const skillBarClasses = 'w-full bg-muted rounded-full h-2.5';
const skillBarFillClasses = 'bg-accent h-2.5 rounded-full';


//skill bars data
const skillBarsData = [
    { skill: "Face Recognition system", percentage: 95 },
    { skill: "Chatbot", percentage: 90 },
    { skill: "IOT System", percentage: 90 },
    { skill: "AI Development", percentage: 85 },
    { skill: "App Development", percentage: 90 },
    { skill: "Robotic", percentage: 90 }
];

// Array of experiences
const experiences = [
    {
        date: "2016 - 2021",
        position: "CEO",
        company: "Hippo Food & Hippo Xpress Delivery"
    },
    {
        date: "2021 - 2023",
        position: "CEO",
        company: "IBS Hippo PLT"
    },
    {
        date: "2023 - Present",
        position: "CEO",
        company: "Hippo Tech Fast"
    }
];



//skill bar component
const SkillBar = ({ skill, percentage, animate }) => {
    const [fillWidth, setFillWidth] = useState(0);
    const targetWidth = percentage;
    const fillRef = useRef(null);

    useEffect(() => {
        if (animate) {
            let currentWidth = 0;
            const animationFrame = () => {
                if (currentWidth < targetWidth) {
                    currentWidth += 1;
                    setFillWidth(currentWidth);
                    requestAnimationFrame(animationFrame);
                }
            };
            requestAnimationFrame(animationFrame);
        }
    }, [animate, targetWidth]);

    useEffect(() => {
        if (!animate) {
            setFillWidth(0);
        }
    }, [animate]);

    return (
        <div>
            <div className="flex justify-between mb-1">
                <span>{skill}</span>
                <span>{animate ? `${fillWidth}%` : `${percentage}%`}</span>
            </div>
            <div className={skillBarClasses}>
                <div
                    ref={fillRef}
                    className={skillBarFillClasses}
                    style={{
                        width: `${fillWidth}%`,
                        transition: 'width 1s ease-in-out',
                        backgroundColor: animate ? `${onlyColor}` : 'initial',
                    }}
                ></div>
            </div>
        </div>
    );
};





//main component
const SkillsAndExperiences = () => {
    const [animateSkills, setAnimateSkills] = useState(false);

    useEffect(() => {
        const handleScroll = () => {
            const skillsSection = document.getElementById('skill');
            if (!skillsSection) return;

            const { top } = skillsSection.getBoundingClientRect();
            const screenHeight = window.innerHeight;

            if (top <= screenHeight && !animateSkills) {
                setAnimateSkills(true);
            }
        };

        window.addEventListener('scroll', handleScroll);
        return () => {
            window.removeEventListener('scroll', handleScroll);
        };
    }, [animateSkills]);


    return (
        <>
            <h2 className="text-4xl md:text-6xl lg:text-6xl font-extrabold text-center mt-20"><span id='name-span'>Skills & Experiences</span></h2>
            <div id='skill' className="my-4 mx-auto w-[100%] md:w-[90%] lg:w-[80%] p-6 bg-white">
                <div id='experience' className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6 md:divide-x md:divide-zinc-300">
                    <div className={cardClass}>
                        <h3 className={fillClass}>Experiences</h3>
                        {experiences.map((experience, index) => (
                            <div key={index} className="mt-4">
                                <p className="text-sm text-zinc-400">{experience.date}</p>
                                <p className={textClass}>{experience.position}</p>
                                <p className={textClasses}>{experience.company}</p>
                            </div>
                        ))}
                    </div>
                    <div id='experience2' className='flex flex-col p-4'>
                        <h3 className={fillClass}>Skills</h3>
                        <div className='w-full md:w-full mt-4'>
                            {skillBarsData.map((skillBar, index) => (
                                <SkillBar
                                    key={index}
                                    skill={skillBar.skill}
                                    percentage={skillBar.percentage}
                                    animate={animateSkills}
                                />
                            ))}
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
};

export default SkillsAndExperiences;
